<?php

// LOAD CONFIG

use App\Lib\Router;
use App\Lib\RouteCollectorProxy;

use App\Controller\{ErrorController, HomeController, OrdersController, StatusController};
use App\Util\{Helper, DeviceManager};
use App\Models\TokenMiddleware;

// AUTOLOAD CORE LIBRARIES
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/Helpers/ConfigManager.php';
require_once __DIR__ . '/../config/config.php';

// CHECK IF MAINTENANCE MODE IS ACTIVE
if (\CONFIG['maintenance'] == true) {

   // DISPLAY MAINTENANCE TEMPLATE
   $maintenance = new App\Helpers\MaintenanceMode(\CONFIG['maintenance']);
   $maintenance->render();
} else {

   // LOAD ROUTER CLASS
   session_start();
   $deviceManager = new DeviceManager;
   $deviceManager->deleteExpiredSessions();

   $router = Router::create();

   // client routes
   $router->group('', function (RouteCollectorProxy $group) {
      $group->get('/', [HomeController::class, 'home']);
      $group->get('/login', [HomeController::class, 'sign_in']);
   });

   // Test Hybrid App
   $router->group('/api/v1/statusbar', function (RouteCollectorProxy $group) {
      $group->post('', [StatusController::class, 'statusbar']);

      $group->get('', function () {
         Helper::jsonResult(['status' => 'ok']);
      });
   });

   // API ROUTES
   $router->group('/api/v1/auth', function (RouteCollectorProxy $group) {
      $group->post('/{section}/refresh', function ($section) {
         (new App\Controller\ApiController)->auth($section, 'refresh');
      });

      $group->post('/restaurant/{action}', function ($action) {
         (new App\Controller\ApiController)->auth('restaurant', $action);
      });

      $group->post('/rider/{action}', function ($action) {
         (new App\Controller\ApiController)->auth('rider', $action);
      });
   });

   $router->group('/api/v1/restaurant', function (RouteCollectorProxy $group) {
      //authenticate with token if user is logged in
      $group->add(new TokenMiddleware());

      // Onboarding
      $group->post('/onboarding/{action}', function ($action) {
         $response = (new App\Models\Restaurant)->setupWizard($action);
         Helper::jsonResult($response);
      });

      // Get Data By Restaurant Id
      $group->get('/{id}/profile', function ($id) {
         $response = (new App\Models\Restaurant)->getData((int) $id, 'profile');
         Helper::jsonResult($response);
      });

      $group->get('/{id}/restaurant-settings', function ($id) {
         $response = (new App\Models\Restaurant)->getData((int) $id, 'restaurant-settings');
         Helper::jsonResult($response);
      });

      $group->get('/{id}/restaurant-cuisine', function ($id) {
         $response = (new App\Models\Restaurant)->getData((int) $id, 'restaurant-cuisine');
         Helper::jsonResult($response);
      });

      $group->get('/{id}/restaurant-payment', function ($id) {
         $response = (new App\Models\Restaurant)->getData((int) $id, 'restaurant-payment');
         Helper::jsonResult($response);
      });

      $group->get('/{id}/restaurant-location', function ($id) {
         $response = (new App\Models\Restaurant)->getData((int) $id, 'restaurant-location');
         Helper::jsonResult($response);
      });

      $group->get('/{id}/restaurant-region', function ($id) {
         $response = (new App\Models\Restaurant)->getData((int) $id, 'restaurant-region');
         Helper::jsonResult($response);
      });

      /////////////////////////////////// RESTAURANT SETTINGS ROUTES /////////

      $group->post('/{action}', function ($action) {
         $response = (new App\Models\Restaurant)->runSetting($action);
         Helper::jsonResult($response);
      });

      /////////////////////////////////// CREATE | DELETE | UPDATE | MARK AS /////////////////////////

      $group->post('/create/{action}', function ($action) {
         $response = (new App\Models\Restaurant)->createData($action);
         Helper::jsonResult($response);
      });

      $group->post('/edit/{action}', function ($action) {
         $response = (new App\Models\Restaurant)->updateData($action);
         Helper::jsonResult($response);
      });

      $group->post('/delete/{action}', function ($action) {
         $response = (new App\Models\Restaurant)->deleteData($action);
         Helper::jsonResult($response);
      });

      /////////////////////////////////// RESTAURANT MENU & DISHES ///////////

      $group->get('/{id}/menu/{action}', function ($id, $action) {
         $response = (new App\Models\Restaurant)->getMenu((int) $id, $action);
         Helper::jsonResult($response);
      });

      $group->get('/{id}/menu/{data}/{action}', function ($id, $data, $action) {
         $response = (new App\Models\Restaurant)->getMenu((int) $id, $action, $data);
         Helper::jsonResult($response);
      });

      $group->get('/{id}/dishes/{action}', function ($id, $action) {
         $response = (new App\Models\Restaurant)->getDishes((int) $id, $action);
         Helper::jsonResult($response);
      });

      $group->get('/{id}/customization/{action}', function ($id, $action) {
         $response = (new App\Models\Restaurant)->getCustomization((int) $id, $action);
         Helper::jsonResult($response);
      });

      $group->get('/{id}/customization/{action}/{data}', function ($id, $action, $data) {
         $response = (new App\Models\Restaurant)->getCustomization((int) $id, $action, $data);
         Helper::jsonResult($response);
      });

      /////////////////////////////////// REVIEWS ////////////////////

      $group->get('/{id}/review/{action}', function ($id, $action) {
         $response = (new App\Models\Restaurant)->getReviews($id, $action);
         Helper::jsonResult($response);
      });

      $group->post('/{id}/review/{action}', function ($id, $action) {
         $response = (new App\Models\Restaurant)->runReviews($id, $action);
         Helper::jsonResult($response);
      });

      /////////////////////////////////// ORDERS /////////////////////////

      $group->group('/orders', function (RouteCollectorProxy $order) {
         // Get Orders
         $order->get('/{action}', function ($action) {
            $response = (new OrdersController)->getOrders($action);
            Helper::jsonResult($response);
         });

         // Get Single Order Details
         $order->get('/details/{order_reference}', function ($order_reference) {
            $response = (new OrdersController)->getOrder($order_reference);
            Helper::jsonResult($response);
         });

         // Set Order Status
         $order->post('/status', function () {
            $postData = Helper::decodePostData();
            $response = (new OrdersController)->setStatus($postData);
            Helper::jsonResult($response);
         });
      });

      $group->group('/status', function (RouteCollectorProxy $status) {
         $status->post('/create', function () {
            $response = (new StatusController)->createStatus();
            Helper::jsonResult($response);
         });
      });

   });

   /////////////////////////////////// LEGAL DATA /////////////////////////

   $router->get('/api/v1/legal/{action}', function ($action) {
      (new App\Controller\LegalController)->render($action);
   });

   $router->get('/cron/process/{action}/{client}', function ($action, $client) {
      $response = (new App\Controller\CronController)->process($action, $client);
      Helper::jsonResult($response ?? []);
   });

   $router->get('/api/tracker/{param}/{action}/{client}', function ($param, $action, $client) {
      $response = (new App\Controller\ApiController)->tracker($param, $action, $client);
      Helper::jsonResult($response ?? []);
   });

   // $router->get('/cron/import/{client}', function ($client) {
   //    $response = (new App\Controller\CronController)->import($client);
   //    Helper::jsonResult($response ?? []);
   // });

   $router->run();
}
